﻿/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 © i-net software 1998-2013
*/
using System;
using System.IO;
using System.Net;
using System.Windows.Forms;

using Inet.Viewer.Resources;

namespace Inet.Viewer
{
    /// <summary>
    /// Custom message box to show messages and errors. Has a mode for showing details, e.g. for the exception StackTrace 
    /// <see cref="ShowDetails"/>
    /// </summary>
    public partial class MyMessageBox : Form
    {
        private bool showDetails;

        /// <summary>
        /// Default Constructor
        /// </summary>
        public MyMessageBox()
        {
            InitializeComponent();
            ShowDetails = false;
            this.lblMessageDetails.Text = strings.MessageDetails;
            this.imgError.Image = Images.error_32;
            this.StartPosition = FormStartPosition.CenterParent;
            wbDetails.Visible = false;
        }

        /// <summary>
        /// Constructor that sets the Properties of this page
        /// </summary>
        /// <param name="message">The message itself</param>
        /// <param name="messageDetails">The details of the message like the Stacktrace</param>
        /// <param name="messageTitle">The title of the dialog</param>       
        public MyMessageBox(string message, string messageDetails, string messageTitle) : this()
        {
            this.Message = message;
            this.MessageDetails = messageDetails;
            this.MessageTitle = messageTitle;
            
        }

        /// <summary>
        /// Property to set if the Message Details are shown or not
        /// </summary>
        public bool ShowDetails
        {
            get 
            {
                return showDetails; 
            }
            set
            {
                showDetails = value;
                UpdateDetails();
            }
        }

        /// <summary>
        /// The Details of this message. For example the stacktrace of an Exception
        /// </summary>
        public string MessageDetails
        {
            set
            {
                if (value != null && value.Contains("<html"))
                {
                    wbDetails.DocumentText = value;
                    wbDetails.Visible = true;
                    txtErrorDetails.Text = string.Empty;
                    txtErrorDetails.Visible = false;
                }
                else
                {
                    txtErrorDetails.Text = value;
                    txtErrorDetails.Visible = true;
                    wbDetails.DocumentText = string.Empty;
                    wbDetails.Visible = false;
                }
            }
        }
        
        /// <summary>
        /// The Title of the Message displayed in the title
        /// </summary>
        public string MessageTitle
        {
            get { return this.Text; }
            set { this.Text = value; }
        }
        
        /// <summary>
        /// A short description of the Message
        /// </summary>
        public string Message
        {
            get { return txtMessage.Text; }
            set { txtMessage.Text = value; }
        }
 
        /// <summary>
        /// Close window with click on OK Button
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnOKClick(object sender, EventArgs e)
        {
            this.Close();
        }

        /// <summary>
        /// Switch showing details
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnDetailsClick(object sender, EventArgs e)
        {
            ShowDetails = !showDetails;
        }

        /// <summary>
        /// Shows or hides the details section depending on  the
        /// <see cref="ShowDetails"/>property
        /// </summary>
        private void UpdateDetails()
        {                      
            lblMessageDetails.Visible = ShowDetails;

            if (showDetails)
            {                
                this.Height = 500;                
                this.btnDetails.Text = strings.LessDetails;
            }
            else
            {
                this.Height = 180;                
                this.btnDetails.Text = strings.MoreDetails;
            }
        }   
     
        /// <summary>
        /// This method creates a new Dialog and fills it with the contents of the Exception
        /// </summary>
        /// <param name="e">The Exception</param>        
        public void Show(Exception e)
        {
            if (e is ViewerException)
            {
                Show((ViewerException)e);
                return;
            }
            this.Show(e.GetType()+ ": " +e.Message, e.ToString(), strings.ErrorMessage);
        }

        /// <summary>
        /// This method creates a new Dialog and fills it with the contents of the ViewerException
        /// </summary>
        /// <param name="ve">The Viewer Exception</param>        
        private void Show(ViewerException ve)
        {
            string messageDetails = ve.ToString();
            if (ve.InnerException != null)
            {
                if (ve.InnerException is WebException)
                {
                    try
                    {
                        WebResponse response = ((WebException)ve.InnerException).Response;
                        if (response != null)
                        {
                            using (Stream dataStream = response.GetResponseStream())
                            {
                                using (StreamReader reader = new StreamReader(dataStream))
                                {
                                    messageDetails = reader.ReadToEnd();
                                }
                            }
                        }
                    }
                    catch (Exception)
                    {
                        // ignore, cannot obtain html content
                    }
                }
            }
            
            if (ve.ServerStackTrace != null)
            {
                messageDetails = ve.ServerStackTrace;
            }
            string message;
            if (ve.ErrorCode > 0)
            {
                message = strings.ErrorCode + ": " + ve.ErrorCode + "\r\n" + ve.Message;
            }
            else
            {
                message = ve.Message;
            }
            this.Show(message, messageDetails, strings.ErrorMessage);
        }

        /// <summary>
        /// This method creates a new Dialog and fills it with the following contents and opens it
        /// </summary>
        /// <param name="message">The message itself</param>
        /// <param name="messageDetails">The details of the message like the Stacktrace</param>
        /// <param name="messageTitle">The title of the dialog</param>
        public void Show(string message, string messageDetails, string messageTitle)
        {
            if (!Visible)
            {
                this.Message = message;
                this.MessageDetails = messageDetails;
                this.MessageTitle = messageTitle;

                ShowDialog();
            }        
        }
    }
}
